 /**************************************************************************************
 
   Copyright (c) Hilscher GmbH. All Rights Reserved.
 
 **************************************************************************************
 
   Filename:
    $Id: ConnectorAPI.cpp 7136 2015-07-22 14:05:18Z Robert $
   Last Modification:
    $Author: Robert $
    $Date: 2015-07-22 16:05:18 +0200 (Mi, 22 Jul 2015) $
    $Revision: 7136 $
   
   Targets:
     Win32/ANSI   : yes
     Win32/Unicode: no
     WinCE        : no
 
   Description:
    Implementation of the netX Connector API
       
   Changes:
 
     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     3         20.07.15    RM       Added error message on store configuration fault
     2         27.11.09    SS       Review
     1         28.07.09    SS       initial version
 
**************************************************************************************/

/*****************************************************************************/
/*! \file TCPConnector/ConnectorAPI.cpp                                      */
/*   netX Connector API implementation                                       */
/*****************************************************************************/

#include "stdafx.h"

#include "TCPLayer.h"
#include "ConnectorAPI.h"
#include "netXConnectorErrors.h"
#include "TCPConfig.h"
#include "TCPConfigDlg.h"
#include <stdio.h>

/*****************************************************************************/
/*! \addtogroup netX_CONNECTOR_TCP netX TCP Connector                        */
/*! \{                                                                       */
/*****************************************************************************/

/* UUID of the TCP Connector */
const UUID g_tConnectorUUID = { 0x1719D5A0, 0xDD3B, 0x48e2, {0xBB, 0xF6, 0xDD, 0xDA, 0xAE, 0xD7, 0xE8, 0xB7} };

/* Global TCP layer object */
CTCPLayer*  g_pcTCPLayer  = NULL;

/* Global object for configuration of TCP layer, TRUE loads config from registry */
CTCPConfig  g_cTCPConfig(TRUE);

/***************************************************************************
* 
*   
*   Connector API Functions
*
*
*
*
*
******************************************************************************/

/*****************************************************************************/
/*! Get identifier and UUID of the connector
*   \param  szIdentifier Returns identifier of the connector
*   \param  pvUUID       Returns UUID of the connector
*   \return NXCON_NO_ERROR on success                                        */
/*****************************************************************************/
long APIENTRY netXConGetIdentifier( char* szIdentifier, void* pvUUID)
{
  if(!szIdentifier)
    return NXCON_DRV_INVALID_POINTER;

  if(!pvUUID)
    return NXCON_DRV_INVALID_POINTER;
  
  /* Copy identifier of the connector */
  strcpy( szIdentifier, TCP_CONNECTOR_IDENTIFIER);

  /* Copy UUID of the connector */
  UUID* ptUUID = (UUID*)pvUUID;
  *ptUUID      = g_tConnectorUUID;
  
  return NXCON_NO_ERROR;
}

/*****************************************************************************/
/*! Open the connector
*   \param  pfnDevNotifyCallback  Function pointer for the callback
*   \param  pvUser                User pointer
*   \return NXCON_NO_ERROR on success                                        */
/*****************************************************************************/
long APIENTRY netXConOpen( PFN_NETXCON_DEVICE_NOTIFY_CALLBACK pfnDevNotifyCallback, void* pvUser)
{
  if(g_pcTCPLayer)
    return NXCON_DRV_WAS_OPENED_BEFORE;

  g_pcTCPLayer  = new CTCPLayer;

  return g_pcTCPLayer->Init( pfnDevNotifyCallback, pvUser);
}

/*****************************************************************************/
/*! Close a open connector 
*   \return NXCON_NO_ERROR on success                                        */
/*****************************************************************************/
long APIENTRY netXConClose(void)
{ 
  long lRet = NXCON_DRV_NOT_INITIALIZED;

  if( g_pcTCPLayer)
  {
    g_pcTCPLayer->Deinit();

    delete g_pcTCPLayer;
    g_pcTCPLayer = NULL;
    
    lRet = NXCON_NO_ERROR;
  }
return lRet;
}

/*****************************************************************************/
/*! Create device interface for the connector
*   \param  szDeviceName    Name of the interface (e.g. TCP0)
*   \return PCONNECTOR_INTERFACE on success                                  */
/*****************************************************************************/
PCONNECTOR_INTERFACE APIENTRY netXConCreateInterface( const char* szDeviceName)
{ 
  if(!g_pcTCPLayer)
    return NULL;

return g_pcTCPLayer->CreateInterface( szDeviceName);
}

/*****************************************************************************/
/*! Start the connector interface
*   \param  pvInterface     Pointer of the interface 
*   \param  pfnReceiveData  Function pointer of the receive callback
*   \param  pvUser          User pointer
*   \return NXCON_NO_ERROR on success                                        */
/*****************************************************************************/
long APIENTRY netXConStartInterface( PCONNECTOR_INTERFACE pvInterface, PFN_NETXCON_DEVICE_RECEIVE_CALLBACK pfnReceiveData, void* pvUser)
{ 
  long lRet = NXCON_NO_ERROR;

  if(!g_pcTCPLayer)
    return NXCON_DRV_NOT_INITIALIZED;

  if(!pvInterface)
    return NXCON_DRV_INVALID_POINTER;

  g_pcTCPLayer->LockInterfaceAccess();
  
  /* Check if interface pointer is valid */
  if (NXCON_NO_ERROR == (lRet = g_pcTCPLayer->ValidateInterface( pvInterface)))
  {
    lRet = reinterpret_cast<CTCPInterface*>(pvInterface)->Start(pfnReceiveData, pvUser);
  } 

  g_pcTCPLayer->UnlockInterfaceAccess();

  return lRet;
}

/*****************************************************************************/
/*! Stop the connector interface 
*   \param  pvInterface   Pointer of the interface 
*   \return NXCON_NO_ERROR on success                                        */
/*****************************************************************************/
long APIENTRY netXConStopInterface(PCONNECTOR_INTERFACE pvInterface)
{ 
  long lRet = NXCON_NO_ERROR;

  if(!g_pcTCPLayer)
    return NXCON_DRV_NOT_INITIALIZED;

  if(!pvInterface)
    return NXCON_DRV_INVALID_POINTER;

  g_pcTCPLayer->LockInterfaceAccess();
  
  /* Check if interface pointer is valid */
  if (NXCON_NO_ERROR == (lRet = g_pcTCPLayer->ValidateInterface( pvInterface)))
  {
    reinterpret_cast<CTCPInterface*>(pvInterface)->Stop();
  }
  
  g_pcTCPLayer->UnlockInterfaceAccess();

  return lRet;
}

/*****************************************************************************/
/*! Send data via the connector interface 
*   \param  pvInterface    Pointer of the interface 
*   \param  pabData        Data pointer 
*   \param  ulDataLen      Length of the send data 
*   \return NXCON_NO_ERROR on success                                        */
/*****************************************************************************/
long APIENTRY netXConSendInterface( PCONNECTOR_INTERFACE pvInterface, unsigned char* pabData, unsigned long ulDataLen)
{ 
  long lRet = NXCON_NO_ERROR;

  if(!g_pcTCPLayer)
    return NXCON_DRV_NOT_INITIALIZED;

  if( !pvInterface  ||
      !pabData        )
    return NXCON_DRV_INVALID_POINTER;

  g_pcTCPLayer->LockInterfaceAccess();
  
  /* Check if interface pointer is valid */
  if (NXCON_NO_ERROR == (lRet = g_pcTCPLayer->ValidateInterface( pvInterface)))
  {
    lRet = reinterpret_cast<CTCPInterface*>(pvInterface)->Send( pabData, ulDataLen);
  } 

  g_pcTCPLayer->UnlockInterfaceAccess();

  return lRet;
}

/*****************************************************************************/
/*! Get the configuration
*   \param  pvConfig     Pointer to result of the request
*   \param  eCmd         eCMD_CONFIG_GETLEN/eCMD_CONFIG_GETSTRING/
*                        eCMD_DIALOG_GETLEN/eCMD_DIALOG_GETSTRING
*   \return NXCON_NO_ERROR on success                                        */
/*****************************************************************************/
long APIENTRY netXConGetConfig( NETX_CONNECTOR_CONFIG_CMD_E eCmd, void* pvConfig)
{ 
  long lRet = NXCON_NO_ERROR;

  if( !pvConfig)
    return NXCON_DRV_INVALID_POINTER;

  g_cTCPConfig.Lock();
  CString csConfig; 
  lRet = g_cTCPConfig.CreateString( csConfig);
  g_cTCPConfig.Unlock();

  if (NXCON_NO_ERROR != lRet)
    return lRet;

  switch (eCmd)
  {
    case eCMD_CONFIG_GETLEN:
      {
        unsigned long* pulLength = (unsigned long*)pvConfig;
        *pulLength = csConfig.GetLength();
      }
      break;

    case eCMD_CONFIG_GETSTRING:
      {
        char* pszString = (char*)pvConfig;
        strcpy(pszString, csConfig.GetBuffer());
      }
      break;
    
    case eCMD_DIALOG_GETLEN:
      {
        CTCPConfigDlg* pcDialog = CTCPConfigDlg::GetInstance();    

        if (pcDialog->GetWindowHandle())
        {
          /* Get configuration string from config object */
          CString csConfig;
          if( NXCON_NO_ERROR == (lRet = pcDialog->GetConfiguration( csConfig)))
          {
            unsigned long* pulLength = (unsigned long*)pvConfig;
            *pulLength = csConfig.GetLength();
          }
        } else
        {
          lRet = NXCON_DRV_NOT_INITIALIZED;
        }
      }
      break;

    case eCMD_DIALOG_GETSTRING:
      {
        CTCPConfigDlg* pcDialog = CTCPConfigDlg::GetInstance();    

        if (pcDialog->GetWindowHandle())
        {
          /* Get configuration string from config object */
          CString csConfig;
          if( NXCON_NO_ERROR == (lRet = pcDialog->GetConfiguration( csConfig)))
          {
            char* pszString = (char*)pvConfig;
            strcpy(pszString, csConfig.GetBuffer());
          }
        } else
        {
          lRet = NXCON_DRV_NOT_INITIALIZED;
        }
      }
      break;

    default:
      lRet = NXCON_DRV_INVALID_PARAMETER;
      break;
  }

  return lRet;
}

/*****************************************************************************/
/*! Set the connector configuration
*   \param  eCmd         eCMD_CONFIG_SETSTRING/eCMD_DIALOG_SETSTRING
*   \param  szConfig     Configuration string
*   \return NXCON_NO_ERROR on success                                        */
/*****************************************************************************/
long APIENTRY netXConSetConfig( NETX_CONNECTOR_CONFIG_CMD_E eCmd, const char* szConfig)
{ 
  long lRet = NXCON_NO_ERROR;

  if( !szConfig)
    return NXCON_DRV_INVALID_POINTER;
  
  switch (eCmd)
  {
    case eCMD_CONFIG_SETSTRING:
      g_cTCPConfig.Lock();
      if (NXCON_NO_ERROR == (lRet = g_cTCPConfig.ParseString( szConfig)))
      {
        /* Store Configuration to registry */
        CString szRegPath;
        long lRet = NXCON_NO_ERROR;
        if( NXCON_NO_ERROR != (lRet = g_cTCPConfig.StoreToRegistry(szRegPath)))
        {
          CString csText;
          csText.Format(_T("Failed to store TCP connector configuration\r<%s>"), szRegPath);
          MessageBox( NULL, csText, _T("netXTransport DLL connector error"), MB_ICONEXCLAMATION);
        }
      }
      g_cTCPConfig.Unlock();

      /* Reinit Connector */
      if(g_pcTCPLayer)
        g_pcTCPLayer->Reinit();
      break;

    case eCMD_DIALOG_SETSTRING:
      {
        CTCPConfigDlg* pcDialog = CTCPConfigDlg::GetInstance();    

        if (pcDialog->GetWindowHandle())
        {
          /* Set configuration string from config object */
          lRet = pcDialog->SetConfiguration( szConfig);
        } else
        {
          lRet = NXCON_DRV_NOT_INITIALIZED;
        }
      }
      break;

    default:
      lRet = NXCON_DRV_INVALID_PARAMETER;
      break;
  }

  return lRet;
}

/*****************************************************************************/
/*! Get connector information
*   \param  eCmd            eSTATE/eUUID/eTYPE/eESTABLISH_TIMEOUT
*   \param  ulSize          Size of information structure
*   \param  pvConnectorInfo Connector information structure
*   \return NXCON_NO_ERROR on success                                        */
/*****************************************************************************/
long APIENTRY netXConGetInformation( NETX_CONNECTOR_INFO_E eCmd, unsigned long ulSize, void* pvConnectorInfo)
{
  if (!pvConnectorInfo)
    return NXCON_DRV_INVALID_POINTER;

  long lRet = NXCON_NO_ERROR;

  switch(eCmd)
  {
    case eSTATE:   
      if(sizeof(NETX_CONNECTOR_INFO_STATE_U) <= ulSize)
      {
        NETX_CONNECTOR_INFO_STATE_U* ptConnectorState = (NETX_CONNECTOR_INFO_STATE_U*)pvConnectorInfo;
        /* Populate Connector Info structure */
        g_cTCPConfig.Lock();
        ptConnectorState->tBfState.fEnable = g_cTCPConfig.IsConnectorEnabled();
        g_cTCPConfig.Unlock();
      }else
      {
        lRet = NXCON_DRV_BUFFER_TOO_SHORT;
      }
    break;

    case eUUID:
      if(sizeof(UUID) <= ulSize)
      {
        UUID* ptUUID = (UUID*)pvConnectorInfo;
        *ptUUID      = g_tConnectorUUID;
      } else
      {
        lRet = NXCON_DRV_BUFFER_TOO_SHORT;
      }
    break;
    
    case eTYPE:  
      if(sizeof(NETX_CONNECTOR_TYPE_U) <= ulSize)
      {
        NETX_CONNECTOR_TYPE_U* ptConnectorType = (NETX_CONNECTOR_TYPE_U*)pvConnectorInfo;
        
        /* Clear Connector Type structure */
        ptConnectorType->ulType = 0;
        /* Populate Connector Type structure */
        ptConnectorType->tBfType.fNXDRV_TYPE_ETHERNET = TRUE;

      }else
      {
        lRet = NXCON_DRV_BUFFER_TOO_SHORT;
      }
    break;
    
    case eIDENTIFIER:
      if(strlen(TCP_CONNECTOR_IDENTIFIER) < ulSize)
      {
        /* Copy identifier of the connector */
        strcpy( (char*)pvConnectorInfo, TCP_CONNECTOR_IDENTIFIER);        
      } else
      {
        lRet = NXCON_DRV_BUFFER_TOO_SHORT;
      }
    break;

    case eLONG_INTERFACE_NAME:
      {
        INTERFACE_NAME_TRANSLATION_T* ptTranslate = (INTERFACE_NAME_TRANSLATION_T*)pvConnectorInfo;
        
        /* Split interface name in Identifier, IP Address and Port */
        std::string szSource( ptTranslate->szSource);
        size_t posSeparator  = szSource.find("_");

        if ( string::npos == posSeparator)
        {
          /* Interface name does not match the pattern */
          lRet = NXCON_DRV_INVALID_PARAMETER;

        } else
        {

          /* Split szInterfaceName into IP address and TCP Port */
          std::string szInterfaceName = szSource.substr( 0, posSeparator);
          std::string szBoardName     = szSource.substr( posSeparator + 1, string::npos - posSeparator);

          CTCPInterface* pcInterface = g_pcTCPLayer->FindInterface(szInterfaceName.c_str());

          if(NULL == pcInterface)
          {
            lRet = NXCON_DRV_INVALID_PARAMETER;
          } else
          {
            unsigned long  ulIpAddress = pcInterface->GetIPAddress();
            unsigned short usPort      = pcInterface->GetTCPPort();
            struct in_addr tIpAddress;

            tIpAddress.S_un.S_addr = ulIpAddress;

            int iResult = _snprintf(ptTranslate->szDestination, 
                                    ptTranslate->ulDestinationSize,
                                    "%u.%u.%u.%u:%u\\%s",
                                    tIpAddress.S_un.S_un_b.s_b1,
                                    tIpAddress.S_un.S_un_b.s_b2,
                                    tIpAddress.S_un.S_un_b.s_b3,
                                    tIpAddress.S_un.S_un_b.s_b4,
                                    ntohs(usPort), szBoardName.c_str());

            if(iResult < 0 )
            {
              lRet = NXCON_DRV_BUFFER_TOO_SHORT;
            } else
            {
              lRet = NXCON_NO_ERROR;
            }
          }
        }
      }
      break;

    case eSHORT_INTERFACE_NAME:
      {
        INTERFACE_NAME_TRANSLATION_T* ptTranslate           = (INTERFACE_NAME_TRANSLATION_T*)pvConnectorInfo;
        ULONG                         aulIpAddr[4]          = {0};
        ULONG                         ulPort                = 0;
        char                          szBoardName[MAX_PATH] = {0};

        if(6 != sscanf(ptTranslate->szSource, "%u.%u.%u.%u:%u\\%s",
                       &aulIpAddr[0], &aulIpAddr[1],
                       &aulIpAddr[2], &aulIpAddr[3],
                       &ulPort, szBoardName))
        {
          lRet = NXCON_DRV_INVALID_PARAMETER;

        } else if( (aulIpAddr[0] > MAXBYTE) ||
                   (aulIpAddr[1] > MAXBYTE) ||
                   (aulIpAddr[2] > MAXBYTE) ||
                   (aulIpAddr[3] > MAXBYTE) ||
                   (ulPort       > MAXWORD) )
        {
          lRet = NXCON_DRV_INVALID_PARAMETER;

        } else
        {
          struct in_addr tIpAddr = {0};
          tIpAddr.S_un.S_un_b.s_b1 = (BYTE)aulIpAddr[0];
          tIpAddr.S_un.S_un_b.s_b2 = (BYTE)aulIpAddr[1];
          tIpAddr.S_un.S_un_b.s_b3 = (BYTE)aulIpAddr[2];
          tIpAddr.S_un.S_un_b.s_b4 = (BYTE)aulIpAddr[3];

          /* Create interface with given ip address and tcp port.If an interface with this 
             configuration already exists, the function returns a pointer reference to the 
             existing interface object */
          CTCPInterface* pcInterface = g_pcTCPLayer->CreateInterface(tIpAddr.S_un.S_addr, htons((USHORT)ulPort));

          if ( NULL == pcInterface )
          {
            /* Failed to create interface with the given configuration */
            lRet = NXCON_DRV_INVALID_PARAMETER;
          } else
          {
            if( NXCON_NO_ERROR == (lRet = pcInterface->GetShortInterfaceName(ptTranslate->ulDestinationSize, ptTranslate->szDestination)) )
              _snprintf( ptTranslate->szDestination, ptTranslate->ulDestinationSize, "%s_%s", ptTranslate->szDestination, szBoardName);     
          }
        }
      }
      break;
    
    case eDESCRIPTION:
      if(strlen(TCP_CONNECTOR_DESCRIPTION) < ulSize)
      {
        /* Copy identifier of the connector */
        strcpy( (char*)pvConnectorInfo, TCP_CONNECTOR_DESCRIPTION);        
      } else
      {
        lRet = NXCON_DRV_BUFFER_TOO_SHORT;
      }
    break;

    default:
      lRet = NXCON_DRV_INVALID_PARAMETER;
      break;

  }
  
  return lRet;
}


/*****************************************************************************/
/*! Get interface information
*   \param  pvInterface     Reference to interface object
*   \param  eCmd            eINTERFACE_STATE/eINTERFACE_NAME/eHUMAN_READABLE_NAME/
*                           eSEND_TIMEOUT/eRESET_TIMEOUT/eKEEPALIVE_TIMEOUT                
*   \param  ulSize          Size of information structure
*   \param  pvInterfaceInfo Interface information structure
*   \return NXCON_NO_ERROR on success                                        */
/*****************************************************************************/
long APIENTRY netXConGetInformationInterface( PCONNECTOR_INTERFACE pvInterface, NETX_INTERFACE_INFO_E eCmd, unsigned long ulSize, void* pvInterfaceInfo)
{
  if(!g_pcTCPLayer)
    return NXCON_DRV_NOT_INITIALIZED;

  if (!pvInterfaceInfo)
    return NXCON_DRV_INVALID_POINTER;

  if(!pvInterface)
    return NXCON_DRV_INVALID_POINTER;

  long lRet = NXCON_NO_ERROR;

  g_pcTCPLayer->LockInterfaceAccess();
  
  if (NXCON_NO_ERROR == (lRet = g_pcTCPLayer->ValidateInterface( pvInterface)))
  {
    CTCPInterface* pcIntf = reinterpret_cast<CTCPInterface*>(pvInterface);      

    switch(eCmd)
    {
      case eINTERFACE_STATE:   
        if(sizeof(unsigned long) <= ulSize)
        { 
          unsigned long* pulInterfaceState = (unsigned long*)pvInterfaceInfo;
          *pulInterfaceState = pcIntf->GetInterfaceState();

        }else
        {
          lRet = NXCON_DRV_BUFFER_TOO_SHORT;
        }
      break;

      case eSEND_TIMEOUT:  
        if(sizeof(unsigned long) <= ulSize)
        {
          unsigned long* pulInterfaceTimeout = (unsigned long*)pvInterfaceInfo;
          *pulInterfaceTimeout =  pcIntf->GetSendTimeout();

        }else
        {
          lRet = NXCON_DRV_BUFFER_TOO_SHORT;
        }
      break;

      case eRESET_TIMEOUT:  
        if(sizeof(unsigned long) <= ulSize)
        {
          unsigned long* pulInterfaceTimeout = (unsigned long*)pvInterfaceInfo;
          *pulInterfaceTimeout =  pcIntf->GetResetTimeout();
        }else
        {
          lRet = NXCON_DRV_BUFFER_TOO_SHORT;
        }
      break;

      case eKEEPALIVE_TIMEOUT:  
        if(sizeof(unsigned long) <= ulSize)
        {
          unsigned long* pulInterfaceTimeout = (unsigned long*)pvInterfaceInfo;
          *pulInterfaceTimeout = pcIntf->GetKeepAliveTimeout();
        }else
        {
          lRet = NXCON_DRV_BUFFER_TOO_SHORT;
        }
      break;

      default:
        lRet = NXCON_DRV_INVALID_PARAMETER;
        break;

    }
  } 

  g_pcTCPLayer->UnlockInterfaceAccess();
  
  return lRet;
}

/*****************************************************************************/
/*! Display configuration dialog
*   \param  hParentWnd Parent Window
*   \param  szConfig   Configuration string to configure dialog
*   \return Handle to dialog window                                          */
/*****************************************************************************/
HWND APIENTRY netXConCreateDialog (HWND hParentWnd, const char* szConfig)
{
  if( !hParentWnd)
    return NULL;

  CTCPConfigDlg*  pcDialog = CTCPConfigDlg::GetInstance();
  HWND            hWnd     = pcDialog->GetWindowHandle();

  if (!hWnd)
  {
    if (szConfig)
    {
      /* Create dialog and use the configuration supplied via config string */
      hWnd = pcDialog->Create( hParentWnd, szConfig);
    
    } else
    {    
      /* No config string supplied. Create dialog without initial config */
      hWnd = pcDialog->Create( hParentWnd);
    }
  }

  return hWnd;
}

/*****************************************************************************/
/*! Remove configuration dialog                                              */
/*****************************************************************************/
void APIENTRY netXConEndDialog ()
{
  CTCPConfigDlg* pcDialog = CTCPConfigDlg::GetInstance();    
  
  if (pcDialog)
  {
    pcDialog->Destroy();
  }
}

/*****************************************************************************/
/*! \}                                                                       */
/*****************************************************************************/

